<?php
require_once __DIR__ . '/../functions.php';
require_role(['admin']);
$db = get_db();
$msg = '';
$err = '';

// load table list for UI
$tables = [];
$tr = $db->query('SHOW TABLES');
while ($r = $tr->fetch_row()) $tables[] = $r[0];

// load classes for UI (used by delete-by-class form)
$classes = $db->query('SELECT id, name FROM classes ORDER BY name')->fetch_all(MYSQLI_ASSOC);

// Handle backup (download of entire DB as .sql)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['backup_db'])) {
  if (!csrf_verify($_POST['csrf'] ?? '')) { $err = 'Invalid CSRF'; }
  else {
    // build SQL dump for selected tables or all
    $selected = [];
    if (!empty($_POST['tables']) && is_array($_POST['tables'])) {
      foreach ($_POST['tables'] as $t) {
        // sanitize table name (allow alphanum and _)
        if (preg_match('/^[a-zA-Z0-9_]+$/', $t)) $selected[] = $t;
      }
    }
    if (empty($selected)) {
      $tables_res = $db->query('SHOW TABLES');
      $selected = [];
      while ($r = $tables_res->fetch_row()) $selected[] = $r[0];
    }

    $sql_dump = "-- ABSENSI DB BACKUP\n-- Generated: " . date('c') . "\n\n";
    foreach ($selected as $table) {
      // CREATE TABLE
      $cr = $db->query('SHOW CREATE TABLE `'.$db->real_escape_string($table).'`');
      $crrow = $cr->fetch_assoc();
      $sql_dump .= "-- Table structure for `$table`\n";
      $sql_dump .= $crrow['Create Table'] . ";\n\n";
      // DATA
      $res = $db->query('SELECT * FROM `'.$db->real_escape_string($table).'`');
      if ($res && $res->num_rows) {
        $sql_dump .= "-- Dumping data for `$table`\n";
        while ($row = $res->fetch_assoc()) {
          $cols = array_map(function($c){ return "`".str_replace("`","``",$c)."`"; }, array_keys($row));
          $vals = array_map(function($v){ if (is_null($v)) return 'NULL'; return "'".addslashes($v)."'"; }, array_values($row));
          $sql_dump .= "INSERT INTO `$table` (".implode(',', $cols).") VALUES (".implode(',', $vals).");\n";
        }
        $sql_dump .= "\n";
      }
    }
    // send as download
    header('Content-Type: application/sql');
    header('Content-Disposition: attachment; filename="absensi_backup_'.date('Ymd_His').'.sql"');
    echo $sql_dump;
    exit;
  }
}

// Handle import (.sql)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['import_db'])) {
    if (!csrf_verify($_POST['csrf'] ?? '')) { $err = 'Invalid CSRF'; }
    elseif (empty($_FILES['sql_file']) || $_FILES['sql_file']['error'] !== UPLOAD_ERR_OK) {
        $err = 'File tidak diupload atau error saat upload.';
    } else {
        $content = file_get_contents($_FILES['sql_file']['tmp_name']);
        if ($content === false) { $err = 'Gagal membaca file.'; }
        else {
            // execute via multi_query
            if ($db->multi_query($content)) {
                // consume all results
                do { if ($res = $db->store_result()) { $res->free(); } } while ($db->more_results() && $db->next_result());
                $msg = 'Import database selesai.';
            } else {
                $err = 'Import gagal: ' . $db->error;
            }
        }
    }
}

// Handle truncate students
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['truncate_students'])) {
    if (!csrf_verify($_POST['csrf'] ?? '')) { $err = 'Invalid CSRF'; }
    else {
        try {
            $db->begin_transaction();
            $db->query('SET FOREIGN_KEY_CHECKS = 0');
            $db->query('TRUNCATE TABLE attendance');
            $db->query('TRUNCATE TABLE students');
            $db->query('SET FOREIGN_KEY_CHECKS = 1');
            $db->commit();
            $msg = 'Tabel students (dan attendance) berhasil dikosongkan.';
        } catch (Exception $e) {
            $db->rollback();
            $err = 'Gagal truncate: ' . $e->getMessage();
        }
    }
}

// Handle truncate classes
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['truncate_classes'])) {
    if (!csrf_verify($_POST['csrf'] ?? '')) { $err = 'Invalid CSRF'; }
    else {
        try {
            $db->begin_transaction();
            $db->query('SET FOREIGN_KEY_CHECKS = 0');
            $db->query('TRUNCATE TABLE attendance');
            $db->query('TRUNCATE TABLE students');
            $db->query('TRUNCATE TABLE classes');
            $db->query('SET FOREIGN_KEY_CHECKS = 1');
            $db->commit();
            $msg = 'Tabel classes, students, dan attendance berhasil dikosongkan.';
        } catch (Exception $e) {
            $db->rollback();
            $err = 'Gagal truncate: ' . $e->getMessage();
        }
    }
}

// Handle delete students by class
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_students_by_class'])) {
  if (!csrf_verify($_POST['csrf'] ?? '')) { $err = 'Invalid CSRF'; }
  else {
    $class_id = (int)($_POST['class_id'] ?? 0);
    if ($class_id <= 0) {
      $err = 'Kelas tidak valid.';
    } else {
      try {
        $db->begin_transaction();
        // delete attendance for students in the class first
        $stmtAtt = $db->prepare('DELETE a FROM attendance a JOIN students s ON a.student_id = s.id WHERE s.class_id = ?');
        $stmtAtt->bind_param('i', $class_id);
        $stmtAtt->execute();
        $stmtAtt->close();

        // delete the students
        $stmtDel = $db->prepare('DELETE FROM students WHERE class_id = ?');
        $stmtDel->bind_param('i', $class_id);
        $stmtDel->execute();
        $affected = $stmtDel->affected_rows;
        $stmtDel->close();

        $db->commit();
        $msg = "Berhasil menghapus $affected siswa dari kelas terpilih.";
      } catch (Exception $e) {
        $db->rollback();
        $err = 'Gagal menghapus siswa: ' . $e->getMessage();
      }
    }
  }
}

?>
<?php require_once __DIR__ . '/_header.php'; ?>
<h4>Pengaturan Database</h4>
<?php if ($msg): ?><div class="alert alert-success"><?= h($msg) ?></div><?php endif; ?>
<?php if ($err): ?><div class="alert alert-danger"><?= h($err) ?></div><?php endif; ?>

<div class="row">
  <div class="col-md-6">
    <div class="card mb-3"><div class="card-body">
      <h5>Backup Database</h5>
      <p>Pilih tabel untuk diekspor (kosong = semua tabel).</p>
      <form method="post">
        <input type="hidden" name="csrf" value="<?=h(csrf_token())?>">
        <div class="mb-2">
          <?php foreach ($tables as $t): ?>
            <label class="form-check form-check-inline">
              <input class="form-check-input" type="checkbox" name="tables[]" value="<?=h($t)?>">
              <span class="form-check-label"><?=h($t)?></span>
            </label>
          <?php endforeach; ?>
        </div>
        <button class="btn btn-primary" name="backup_db">Download Selected Tables (.sql)</button>
      </form>
    </div></div>

    <div class="card mb-3"><div class="card-body">
      <h5>Import Database (Chunked)</h5>
      <p>Unggah file SQL (.sql). File akan diproses secara bertahap dengan progress.</p>
      <form id="uploadForm" enctype="multipart/form-data">
        <input type="hidden" name="csrf" value="<?=h(csrf_token())?>">
        <input type="file" name="sql_file" id="sql_file" accept=".sql" class="form-control mb-2" required>
        <div class="mb-2">
          <button type="button" id="startUpload" class="btn btn-warning">Upload & Start Import</button>
        </div>
      </form>
      <div id="importProgress" style="display:none">
        <div class="mb-2">Progress: <span id="importStatus">0</span></div>
        <div class="progress">
          <div id="importBar" class="progress-bar" role="progressbar" style="width:0%"></div>
        </div>
      </div>
      <pre id="importLog" style="max-height:200px;overflow:auto;background:#fafafa;padding:8px;border:1px solid #eee;margin-top:8px;display:none"></pre>
      <script>
      (function(){
        const uploadBtn = document.getElementById('startUpload');
        const fileInput = document.getElementById('sql_file');
        const progressEl = document.getElementById('importProgress');
        const bar = document.getElementById('importBar');
        const status = document.getElementById('importStatus');
        const log = document.getElementById('importLog');

        uploadBtn.addEventListener('click', function(){
          if (!fileInput.files.length) { alert('Pilih file .sql terlebih dahulu'); return; }
          const f = fileInput.files[0];
          const fd = new FormData();
          fd.append('sql_file', f);
          fd.append('csrf', document.querySelector('input[name="csrf"]').value);
          uploadBtn.disabled = true; uploadBtn.textContent = 'Uploading...';
          fetch('import_upload.php', { method:'POST', body: fd }).then(r=>r.json()).then(j=>{
            if (!j.ok) { alert('Upload gagal: '+(j.error||'unknown')); uploadBtn.disabled=false; uploadBtn.textContent='Upload & Start Import'; return; }
            // start processing
            progressEl.style.display='block'; log.style.display='block';
            const totalSize = j.size; let offset=0; const file = j.file; let done=false;
            function step(){
              fetch('import_process.php?csrf='+encodeURIComponent(document.querySelector('input[name="csrf"]').value)+
                    '&file='+encodeURIComponent(file)+'&offset='+offset+'&statements=30')
                .then(r=>r.json()).then(js=>{
                  if (!js.ok) { log.innerText += 'Error: '+(js.error||'unknown')+'\n'; uploadBtn.disabled=false; uploadBtn.textContent='Upload & Start Import'; return; }
                  offset = js.offset;
                  log.innerText += 'Processed: '+js.processed + (js.done? ' (done)':'') +'\n';
                  // estimate progress by bytes
                  const percent = Math.min(100, Math.round((offset/totalSize)*100));
                  bar.style.width = percent + '%'; status.innerText = percent + '%';
                  if (!js.done) setTimeout(step, 200);
                  else { uploadBtn.disabled=false; uploadBtn.textContent='Upload & Start Import'; alert('Import selesai'); }
            }).catch(e=>{ log.innerText += 'Fetch error: '+e+'\n'; uploadBtn.disabled=false; uploadBtn.textContent='Upload & Start Import'; });
            }
            step();
          }).catch(e=>{ alert('Upload error: '+e); uploadBtn.disabled=false; uploadBtn.textContent='Upload & Start Import'; });
        });
      })();
      </script>
    </div></div>
  </div>

  <div class="col-md-6">
    <div class="card mb-3"><div class="card-body">
      <h5>Truncate Siswa</h5>
      <p>Hapus semua data siswa dan rekap absensi (attendance). Tidak dapat dibatalkan.</p>
      <form id="truncateStudentsForm" method="post">
        <input type="hidden" name="csrf" value="<?=h(csrf_token())?>">
        <input type="hidden" name="truncate_students" value="1">
        <button class="btn btn-danger" name="truncate_students">Truncate Students</button>
      </form>
    </div></div>

    <div class="card mb-3"><div class="card-body">
      <h5>Truncate Kelas</h5>
      <p>Hapus semua data kelas, siswa, dan rekap absensi. Tidak dapat dibatalkan.</p>
      <form id="truncateClassesForm" method="post">
        <input type="hidden" name="csrf" value="<?=h(csrf_token())?>">
        <input type="hidden" name="truncate_classes" value="1">
        <button class="btn btn-danger" name="truncate_classes">Truncate Classes</button>
      </form>
    </div></div>

    <div class="card mb-3"><div class="card-body">
      <h5>Hapus Siswa berdasarkan Kelas</h5>
      <p>Pilih kelas untuk menghapus semua siswa pada kelas tersebut (termasuk data absensi mereka). Tindakan ini tidak dapat dibatalkan.</p>
      <form id="deleteByClassForm" method="post">
        <input type="hidden" name="csrf" value="<?=h(csrf_token())?>">
        <input type="hidden" name="delete_students_by_class" value="1">
        <div class="mb-2">
          <select name="class_id" class="form-select" required>
            <option value="">-- Pilih Kelas --</option>
            <?php foreach ($classes as $c): ?>
              <option value="<?=h($c['id'])?>"><?=h($c['name'])?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <button type="submit" class="btn btn-danger" name="delete_students_by_class">Hapus Siswa di Kelas</button>
      </form>
      <script>
      (function(){
        var f = document.getElementById('deleteByClassForm');
        if (!f) return;
        f.addEventListener('submit', function(ev){
          ev.preventDefault();
          var sel = f.querySelector('select[name="class_id"]');
          var className = sel && sel.options[sel.selectedIndex] ? sel.options[sel.selectedIndex].text : '';
          if (!sel || !sel.value) {
            Swal.fire({icon:'warning', title:'Pilih kelas', text:'Silakan pilih kelas terlebih dahulu.'});
            return;
          }
          Swal.fire({
            title: 'Hapus semua siswa dari "' + className + '"?',
            text: 'Semua siswa pada kelas ini dan rekap absensi mereka akan dihapus. Tindakan ini tidak dapat dibatalkan.',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonText: 'Ya, hapus',
            cancelButtonText: 'Batal'
          }).then(function(result){
            if (result.isConfirmed) {
              // optionally show a loading dialog while submitting
              Swal.fire({title:'Menghapus...', allowOutsideClick:false, didOpen: function(){ Swal.showLoading(); }});
              // submit the form
              f.submit();
            }
          });
        });
      })();
      </script>
    </div></div>
  </div>
</div>

<script>
(function(){
  function attachConfirm(formId, title, text){
    var f = document.getElementById(formId); if (!f) return;
    f.addEventListener('submit', function(ev){
      ev.preventDefault();
      if (typeof Swal === 'undefined'){
        if (!confirm(text)) return; f.submit(); return;
      }
      Swal.fire({
        title: title,
        text: text,
        icon: 'warning',
        showCancelButton: true,
        confirmButtonText: 'Ya, lanjutkan',
        cancelButtonText: 'Batal',
        confirmButtonColor: '#d33'
      }).then(function(res){ if (!res.isConfirmed) return; Swal.fire({title:'Memproses...', allowOutsideClick:false, didOpen:function(){ Swal.showLoading(); }}); f.submit(); });
    });
  }
  attachConfirm('truncateStudentsForm', 'Kosongkan Siswa?', 'Anda yakin ingin mengosongkan tabel siswa dan attendance? Tindakan ini tidak dapat dibatalkan.');
  attachConfirm('truncateClassesForm', 'Kosongkan Kelas?', 'Anda yakin ingin mengosongkan tabel classes, students, dan attendance? Tindakan ini tidak dapat dibatalkan.');
})();
</script>

<?php require_once __DIR__ . '/_footer.php'; ?>
